# Automatically generate and apply RingCentral QoS policy
#
# This is NOT a firewall policy generator.  It is strictly for applying proper DSCP tags to real-time and signaling traffic.
#
#####################################################################################################################################################################################
# DO NOT ALTER THIS SECTION OF CODE # DO NOT ALTER THIS SECTION OF CODE # DO NOT ALTER THIS SECTION OF CODE # DO NOT ALTER THIS SECTION OF CODE # DO NOT ALTER THIS SECTION OF CODE #
# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv #
#
# Self-elevate the script if required
if (-Not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] 'Administrator')) {
 if ([int](Get-CimInstance -Class Win32_OperatingSystem | Select-Object -ExpandProperty BuildNumber) -ge 6000) {
  Write-Host 'Privilege Elevation required.  You may press enter and have this script'
  WRite-Host 'attempt to elevate automatically or press Ctrl-C and run as Administrator.'
  Write-Host ''
  Write-Host 'Press enter to proceed or Ctrl-C to abort'
  Read-Host 
  $CommandLine = "-File `"" + $MyInvocation.MyCommand.Path + "`" " + $MyInvocation.UnboundArguments
  Start-Process -FilePath PowerShell.exe -Verb Runas -ArgumentList $CommandLine
  Exit
 } else {
  Write-Host 'Privilege Elevation required and cannot be done automatically.'
  Write-Host '-- Run again from Administrator account.'
  Exit 2
 }
}
#
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ ##
##  END OF DO NOT ALTER CODE BLOCK  ##  END OF DO NOT ALTER CODE BLOCK  ##  END OF DO NOT ALTER CODE BLOCK  ##  END OF DO NOT ALTER CODE BLOCK  ##  END OF DO NOT ALTER CODE BLOCK ##
#####################################################################################################################################################################################
#
#
$global:codeRev = "20251111"
$global:rc_revision = '20240717'
#
#@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# Function Definitions

# Ask for string answer (with default) and confirm with y/n.  Returns string value entered."
Function getStringWithConfirm {
    param ($DefaultValue,$Prompt)

    $conf_f = 'n'
    while ($conf_f -ne 'y') {
        Write-Host -NoNewline "$Prompt (Default: `"$DefaultValue`") "
        $x = Read-Host 
        if ($x -eq '' -or $x -eq $null) {
            $x = $DefaultValue
        }
        Write-Host -NoNewline "   > Confirm value `"$x`" (y/n)"
        [string]$conf_f = Read-Host
        if ($conf_f.Length) {
            $conf_f = $conf_f.Substring(0,1).ToLower()
        }
    }
    Write-Host ' '
    return $x
}

# Ask y/n.  Returns 0 for no, 1 for yes."
Function getFlag {
    param ($DefaultValue,$Prompt)

    $conf_f = 'x'
    $dv = 'n'
    if ($DefaultValue) { $dv='y' }
    while ($conf_f -ne 'y' -and $conf_f -ne 'n') {
        Write-Host -NoNewline "$Prompt [y/n] (Default value is `"$dv`") "
        [string]$conf_f = Read-Host
        if ($conf_f -eq '' -or $conf_f -eq $null) {
            $conf_f = $dv
        }
        $conf_f = $conf_f.Substring(0,1).ToLower()
    }
    $x = 0
    switch ($conf_f) {
        ('y') { $x = $true; break }
        ('n') { $x = $false; break }
    }
    return $x
}

Function checkGpoExists {
    param ($Name)

    $save_EAP = $ErrorActionPreference
    $ErrorActionPreference = 'silentlycontinue'
    $x = Get-GPO -Name $Name -ErrorAction SilentlyContinue
    $ErrorActionPreference = $save_EAP
    if ($x -eq '' -or $x -eq $null) {
        return [int]0
    } else {
        return [int]1
    }
}

# Insert entry into rules hash
Function addListEntry {
    param($Dscp,$ProgName,$Protocol,$DstPortRange,$DstIpPfx,$DstIpLen,$KeyName)

    $x = $global:ListEntry.Add("$Dscp|$ProgName|$Protocol|$DstPortRange|$DstIpPfx|$DstIpLen|$KeyName")
}

# Break address/mask into components.  Return 2 element array [0] is ip, [1] is mask value.
Function getIpData {
    param($AddrMask)

    [array]$x = $AddrMask.split("/")
    return [array]$x
}

Function getDscpData {
    param($DscpVal)

    [array]$x = $DscpVal.split("|")
    return [array]$x
}

Function getPortData {
    param($PortVal)

    [array]$x = $PortVal.split("\/")
    return [array]$x
}

#@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# Code starts here
#
#@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

Write-Host "Windows RingCentral Policy generator - Rev $global:codeRev"
Write-Host "   RC Definitions:     $global:rc_revision"
Write-Host ' '
Write-Host 'This is NOT a firewall policy generator.  It only applies DSCP markings to real-time'
Write-Host 'and signaling traffic for use by upstream devices for QoS purposes.'
Write-Host ' '
Write-Host 'RingCentral provides this software to you "as is" and without warranty of any kind, '
Write-Host 'express, statutory, implied or otherwise, including without limitation any warranty of '
Write-Host 'merchantability, fitness for a particular purpose or infringement.  No oral or written '
Write-Host 'information or advice given to you by any RingCentral employee, representative or '
Write-Host 'distributor will create a warranty for the software, and you may not rely on any such '
Write-Host 'information or advice.'
Write-Host ' '

# Flags (debug_f - Generate debug printouts)
$global:debug_f = $false
$global:debug_f = getFlag -DefaultValue $global:debug_f -Prompt 'Debug ?'

# Set Flag to $true if RingCentral Meetings legacy product is in use, $false otherwise
$global:enaRCM = $false
$global:enaRCM = getFlag -DefaultValue $global:enaRCM -Prompt 'Generate rules to support RingCentral Meetings obsolete legacy product ? '

Write-Host " "
Write-Host "You may mark all otherwise unmarked RingCentral.exe originated traffic and/or"
Write-Host "traffic egressing to the RingCentral DataCenter locations with a valeu of AF21 (18)"
Write-Host "to ease troubleshooting.  This has no impact on QoS unless you hav defined AF21 for"
Write-Host "other purpose."
Write-Host " "

# Set flag to $true to mark ALL extraneous traffic to DataCenters and/or generated by RingCentral.exe with DSCP 18 (AF21)
$global:markDefault = $false
$global:markDefault = getFlag -DefaultValue $global:markDefault -Prompt 'Mark remaining traffic with AF21(18) ?'

Write-Host " "
Write-Host "Generate LOCAL POLICY regardless of domain membership."
Write-Host " "

# Set flag to $true to ignore domain membership and generate local policy rather than group policy
$global:forceLocalDefault = $false
$global:forceLocalDefault = getFlag -DefaultValue $global:forceLocalDefault -Prompt 'Force LOCAL POLICY regardless of domain status ?'

# Application definitions are limited as to ONLY those functions possible using Windows
#
# RingCentral application definitions
$global:rc_addrs = @("66.81.240.0/20","80.81.128.0/20","103.44.68.0/22","103.129.102.0/23","104.245.56.0/21","185.23.248.0/22","192.209.24.0/21","199.68.212.0/22","199.255.120.0/22","208.87.40.0/22")
$global:rc_dscp_lists = @{
    '46' = "UDP\20000:64999"
    '34' = "UDP\10001:10010"
    '26' = "*\5090:5099|TCP\8083:8090"
}

#@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# Any alterations to code below this line MUST be coordinated with Tim McKee in
# the Customer Engineering group.   mailto:tim.mckee@ringcentral.com
#
#

# Add delay in msec after each command to avoid overruns
$global:delayMsec = 150
$global:retryCount = 9

if ($global:enaRCM) {
    $global:rc_dscp_lists['46'] += "|UDP\8803"
    $global:rc_dscp_lists['34'] += "|*\8801:8802"
}
if ($global:markDefault) {
    $global:rc_dscp_lists['18'] = "*\*"
}

if ($global:debug_f) {
    Write-Host "-> addr: $global:rc_addrs"
    foreach($dscp in @('46','34','26','18')) {
        if ($global:rc_dscp_lists.ContainsKey($dscp)) {
            $val = $global:rc_dscp_lists[$dscp]
            Write-Host "-> dscp-ports[$dscp]: $val"
        }
    }
    Write-Host " "
}

# Determine if Domain based GPO generation or NetQosPolicy generation is required
# Set Policy Name and QoS policy key prefix

$global:domainBased_f = $false
if (((Get-CimInstance -ClassName win32_computersystem).partofdomain -eq $true) -and ($global:forceLocalDefault -eq $false)) {
    $global:domainRole = Get-WmiObject -Class Win32_ComputerSystem | Select-Object -ExpandProperty DomainRole
    if (-not($global:domainRole -match '4|5')) {
        Write-Host ' '
        Write-Host '################ E R R O R ##################'
        Write-Host '## Must be executed on a domain controller ##'
        Write-Host '################ E R R O R ##################'
        Write-Host ' '
        exit 3
    }
    Import-Module ActiveDirectory,GroupPolicy

    $global:domainBased_f = $true
    $global:gpoName = 'RC-QoS'
    $global:keyPrefix = 'HKLM\SOFTWARE\Policies\Microsoft\Windows\QoS'
    $global:domainName = (Get-CimInstance -ClassName win32_computersystem).Domain

    $global:gpoName = getStringWithConfirm -DefaultValue $global:gpoName -Prompt 'Enter GPO Name'
    
    Write-Host '-- This is a domain based system.  The NetQosPolicy will be generated'
    Write-Host "-- directly into GPO `"$global:gpoName`" for domain `"$global:domainName`"."
    Write-Host ' '

    Write-Host "Do NOT change the default KeyLocation without *extensive* technical knowledge and a VERY good reason."
    $global:keyPrefix = getStringWithConfirm -DefaultValue $global:keyPrefix -Prompt 'Key location without trailing backslash'


    # ################# Create GPO to contain the QoS policy or update existing GPO. #################
    $testGPO = Get-GPO -Name $global:gpoName -ErrorAction SilentlyContinue

    if ($?) {
        $testGPO.Description = "QoS/DSCP markings for RingCentral traffic. Rev $global:rc_revision"
        Write-Host "GPO $global:gpoName already exists."
    } else {
        New-GPO -Name $gpoName -Comment "QoS/DSCP markings for RingCentral traffic. Rev $global:rc_revision"
        Write-Host "GPO $gpoName has been created."
    }
    Write-Host ' '
} else {
    if ($global:forceLocalDefault -eq $true) {
        Write-Host '-- The option to FORCE LOCAL POLICY has been activated.'
    } else {
        Write-Host '-- This is a standalone system.'
    }
    Write-Host ' '
    Write-Host '-- A NetQosPolicy will be generated'
    Write-Host '-- for this machine only.'
    Write-Host ' '
}

# Set Flag to $true to ONLY delete existing rules, $false for complete action
$global:onlyDelete = getFlag -DefaultValue $false -Prompt 'Remove rules (Cleanup) ONLY? ' 

Write-Host ' '
Write-Host '####### '
Write-Host 'Summary of special actions to be taken:'
Write-Host ' '
if ($global:debug_f) {
    Write-Host 'Debug enabled'
}
if ($global:enaRCM) {
    Write-Host 'Rules for RingCentral Meetings will be processed'
}
if ($global:onlyDelete) {
    Write-Host 'Rules will be deleted only - nothing will be generated'
}
Write-Host ' '
Write-Host 'Press Enter/Return to proceed'
$x = Read-Host 
Write-Host ' '

#
# Delete existing rules
$matchval = 'RC-*'

$save_EAP = $ErrorActionPreference
$ErrorActionPreference = 'silentlyContinue'
if ($global:domainBased_f -ne 0) {
    $matchval = "*\" + $matchval
    $keynames = $null;
    [array]$keynames = Get-ChildItem -Path Registry::$global:keyPrefix
    Write-Host "Delete existing QoS policy elements matching `"$matchval`""
    Start-Sleep -Milliseconds $global:delayMsec 
    foreach ($i in $keynames) {
        if ($i -like $matchval) {
            Write-Host "...$i...deleting   `r "
            #
            $StopLoop = $false
            [int]$Retrycount = 0
            do {
                try {
                    Remove-GPRegistryValue -Name $gpoName -Key $i -ErrorAction Stop -InformationAction SilentlyContinue  | Out-Null
                    $StopLoop = $true
                }
                catch {
                    if ($Retrycount -gt $global:retryCount) {
                        Write-Host " >>> Unable to remove $i"
                        Write-Host "An error occurred:"
                        Write-Host "Message: $($_.Exception.Message)"
                        Write-Host "Category: $($_.CategoryInfo.Category)"
                        Write-Host "Fully Qualified Error ID: $($_.FullyQualifiedErrorId)"
                        Write-Host "Error Record: $_"
                        $StopLoop = $true
                    } else {
                        Write-Host " --- Issue removing $i, waiting $global:delayMsec msec to retry."
                        Start-Sleep -Milliseconds $global:delayMsec
                    }
                }
            } while ($StopLoop -eq $false)
        }
    }
    Write-Host '                                                         '
    Write-Host ' '
} else {
    Write-Host "......deleting NetQosPolicy `"$matchval`""
    Remove-NetQosPolicy -Name $matchval -Confirm:$false -ErrorAction Continue -InformationAction SilentlyContinue | Out-Null
    Start-Sleep -Milliseconds $global:delayMsec
    Write-Host ' '
}
$ErrorActionPreference = $save_EAP

# Exit if only delete action desired.
if ($global:onlyDelete) {
    Write-Host ' '
    Write-Host 'Press Enter/Return to proceed'
    $x = Read-Host 
    exit
}

# Initialize empty Hash Array in which to build rules list
[System.Collections.ArrayList]$global:ListEntry = @()

# Build internal table in memory
# - Loop through possible DSCP values in order 46(EF), 34(AF41), 26(AF31)
$groupcnt = 0
foreach($dscp in @('46','34','26','18')) {
    $groupcnt = $groupcnt + 1
    #
    # Generate RingCentral rules
    $npfx = "RC-QoS-" + $groupcnt.ToString('00')
    if ($rc_dscp_lists.ContainsKey($dscp)) {
        $val = $global:rc_dscp_lists[$dscp]

        [array]$ports = getDscpData -DscpVal $val
        $po_index = 0;
        foreach($dscpport in $ports) {
            $po_index = $po_index + 1
            [array]$ProtoPort = getPortData -PortVal $dscpport
            $protocol = $ProtoPort[0]
            $portrange = $ProtoPort[1]
            $ad_index = 0
            foreach($IpAddress in $global:rc_addrs) {
                if ($global:debug_f) { Write-Host -NoNewline "$dscp : $dscpport : $IpAddress         `r " }
                [array]$x = getIpData($IpAddress)
                $ad_index = $ad_index + 1
                $xxpfx = $npfx + "-" + $po_index.ToString('00') + "-" + $ad_index.ToString('00')
                addListEntry -Dscp $dscp -ProgName "*" -Protocol $protocol -DstPortRange $portrange -DstIpPfx $x[0] -DstIpLen $x[1] -KeyName $xxpfx
            }
        }
    }
}
Write-Host '                                                 '
Write-Host ' '

# Generate miscellaneous RingCentral rules
if ($global:enaRCM -ne 0) {
    if ($global:debug_f) { Write-Host "-> Adding RCM rules (98)" }
    addListEntry -Dscp 34 -ProgName "RingCentralMeetings.exe" -Protocol "UDP" -DstPortRange "8850:8869" -DstIpPfx "*" -DstIpLen "*" -Keyname "RC-QoS-98-01"
    addListEntry -Dscp 18 -ProgName "RingCentralMeetings.exe" -Protocol "*" -DstPortRange "*" -DstIpPfx "*" -DstIpLen "*" -KeyName "RC-QoS-98-02"
}

if ($global:markDefault -eq $true) {
    if ($global:debug_f) { Write-Host "-> Adding default marking rules (99)" }
    addListEntry -Dscp 18 -ProgName "RingCentral.exe" -Protocol "*" -DstPortRange "*" -DstIpPfx "*" -DstIpLen "*" -KeyName "RC-QoS-99-01"
}

if ($global:domainBased_f) {
    Write-Host "Set `"Do Not Use NLA`""
    Set-GPPrefRegistryValue -Name $global:gpoName -Context Computer -Key "HKLM\SYSTEM\CurrentControlSet\Services\Tcpip\QoS" `
        -ValueName "Do not use NLA" -Value "1" -Type String -Action Update | Out-Null
    Start-Sleep -Milliseconds $global:delayMsec
}

Write-Host 'Insert QoS rules into policy...'
$linecnt = 0
foreach ($line in $global:ListEntry) {
    $linecnt = $linecnt + 1
    [array]$x = $line.split("|")
    $dscp = $x[0]
    $progname = $x[1]
    $protocol = $x[2]
    $dstport = $x[3]
    $dstipaddr = $x[4]
    $dstiplen = $x[5]
    $keyname = $x[6]

    $prec = "129"
    switch ($dscp) {
        46 { $prec = 240; break }
        34 { $prec = 200; break }
        26 { $prec = 160; break }
    }

    Write-Host "...inserting $keyname Dscp=$dscp  Prec=$prec  Prog=$progname $protocol\$dstport`:$dstipaddr/$dstiplen `r "
    if ($global:domainBased_f) {
        #
        $StopLoop = $false
        [int]$Retrycount = 0
        do {
            try {
                Set-GPRegistryValue -Name $global:gpoName -Key "$global:keyPrefix\$keyname" `
                    -ValueName "Version", "Application Name", "Protocol", "Local Port", "Local IP", "Local IP Prefix Length","Remote Port", "Remote IP", "Remote IP Prefix Length", "DSCP Value", "Throttle Rate", "Precedence" `
                    -Type String -Value "1.0", $progname, $protocol, "*", "*", "*", $dstport, $dstipaddr, $dstiplen, $dscp, "-1", $prec | Out-Null
                $StopLoop = $true
            }
            catch {
                if ($Retrycount -gt $global:retryCount) {
                    Write-Host " >>> Unable to add $i"
                    Write-Host "An error occurred:"
                    Write-Host "Message: $($_.Exception.Message)"
                    Write-Host "Category: $($_.CategoryInfo.Category)"
                    Write-Host "Fully Qualified Error ID: $($_.FullyQualifiedErrorId)"
                    Write-Host "Error Record: $_"
                    $StopLoop = $true
                } else {
                    Write-Host " --- Issue adding $i, waiting $global:delayMsec msec to retry."
                    Start-Sleep -Milliseconds $global:delayMsec
                    $Retrycount = $Retrycount + 1
                }
            }
        } while ($StopLoop -eq $false)
    } else {
        $params = @{'Name' = $keyname;
            'DSCPAction' = $dscp;
            'Precedence' = $prec}
                
        [string]$targetIp = $dstipaddr + "/" + $dstiplen
        if ($targetIp -ne '*/*') {
            $params['IPDstPrefixMatchCondition']=$targetIp
        }
            
        if ($protocol -ne '*') {
            $params['IPProtocolMatchCondition']=$protocol
        }
            
        if ($dstport -ne '*') {
            if ($dstport.Contains(':')) {
                [array]$ports = $dstport.split(':')
                [uint16]$stPort = $ports[0]
                [uint16]$enPort = $ports[1]
                $params['IPDstPortStartMatchCondition']=$stPort
                $params['IPDstPortEndMatchCondition']=$enPort
            } else {
                $params['IPDstPortMatchCondition']=$dstport
            }
        }
        New-NetQosPolicy @params | Out-Null
        Start-Sleep -Milliseconds $global:delayMsec
    }
}
Write-Host '                                                                          '
Write-Host 'Completed!!'
Write-Host ' '
Write-Host 'Press Enter/Return to continue.'
$x = Read-Host

# Revision History
#
# 2021-02-11 Initial Beta Release of rev 20210210
# 2021-02-22 Add verbiage re priv elevation
# 2021-02-23 Add configurable post command delay
# 2021-02-25 Add disclaimer for release
# 2022-11-30 Change rules and rework code
# 2025-07-09 Add retry loops